/* 
 * CSim - Discrete system simulation library
 * 
 * Copyright (C) 2002 Vikas G P <vikasgp@users.sourceforge.net>
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 */

/* file: queue.c */

#include <stdlib.h>

#include "csim.h"
#include "queue.h"
#include "util.h"

/* 
 * Inserts a node in it's proper place in the sorted queue, based on the
 * key
 * The function greater(a,b)
 * 	= 1 if a > b
 * 	= 0 otherwise
 */
int queue_node_insert_sorted(CsQueue *queue, 
		       CsQueueNode *node,
		       int(*greater)(void *, void *)
		      )
{
	CsQueueNode *i;
	if( (queue->head) != NULL){ /* Non -empty queue */
		i = queue->tail;
		while( (*greater)(node->key, i->key) ){
			i = i->backward;
		}
		node->forward = i->forward;
		node->backward = i;
		(node->forward)->backward = node;
		i->forward = node;
	}
	else{			   /* Empty queue */
		queue->head = node;
		queue->tail = node;
	}
	return 0;
}


/* Removes the list head from the queue, and returns a pointer to it */ 
CsQueueNode *queue_node_pop(CsQueue *queue)
{
	CsQueueNode *old_head = queue->head;
	if(old_head != NULL){  /* Non-empty queue */
		queue->head = old_head->forward;
		if(queue->head != NULL){
			queue->head->backward = NULL;
		}
		return old_head;
	}
	else{		      /* Empty queue */
		return old_head;
	}
}

/* Creates a new queue node */
CsQueueNode *queue_node_new(void)
{
	CsQueueNode *node;
 	node = (CsQueueNode *)xmalloc(sizeof(CsQueueNode)); 
	node->backward = NULL;
	node->forward = NULL;
	node->key = 0;
	node->data = NULL;
	return node;
}

/* Appends a node to a queue */
void queue_node_append(CsQueue *queue,CsQueueNode *node)
{
	if(queue->tail != NULL){ /* Non-empty queue */
		(queue->tail)->forward = node;
		node->backward = queue->tail;
		node->forward = NULL;
		queue->tail = node;
	}
	else{ 			/* Empty queue */
		queue->head = node;
		queue->tail = node;
	}
		
}

/* Creates a new queue */
CsQueue *queue_new(void)
{
	CsQueue *queue;
	queue = (CsQueue *)xmalloc(sizeof(queue));
	queue->head = NULL;
	queue->tail = NULL;
	return queue;
}

/* Destroys a queue */
int queue_destroy(CsQueue *queue)
{
	CsQueueNode *i = queue->head;
	while( i != NULL ){
		i = i->forward;
		free(i->backward);
	}
	free(queue);
	return 0;
}

/* Destroys a node */
int queue_node_destroy(CsQueueNode *node)
{
	free(node);
	return 0;
}

/* Prepends a node to a queue */
void queue_node_prepend(CsQueue *queue, CsQueueNode *node)
{
	
	if( queue->head != NULL ){	/* Non empty */
		node->forward = queue->head;
		node->backward = NULL;
		node->forward->backward = node;
		queue->head = node;
	}
	else{	/* Empty */
		queue->head = node;
		queue->tail = node;
	}
}

		
		
