/* 
 * CSim - Discrete system simulation library
 * 
 * Copyright (C) 2002 Vikas G P <vikasgp@users.sourceforge.net>
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 */

#include "csim.h"
#include "event.h"
#include "util.h"
#include "queue.h"

extern simtime_t current_time;
extern CsQueue *event_queue;
/* 
 * Creates a new event, and returns it. Argument is the event routine to handle
 * the event
 */

CsEvent *cs_event_new(int (*event_routine)(void *user_data))
{
	CsEvent *event;
	event = (CsEvent *)xmalloc(sizeof(CsEvent)) ;
	event->event_routine = event_routine ;
	event->count = 0;
	return event;
}

/* Returns the count of an event, the number of times it has been executed */

int cs_event_get_count(CsEvent *event)
{
	return event->count;
}

/* Creates a new event notice - internal use only */
static CsEventNotice *event_notice_new(void)
{
	CsEventNotice *notice;
	notice = (CsEventNotice *)xmalloc(sizeof(CsEventNotice));
	return notice;
}

/* 
 * Numerical comparision function, needed for queue_node_insert_sorted 
 * Returns
 *       1 if a > b
 *       0 otherwise
 */
static int num_greater(void *a, void *b)
{
	return ( *((int *)a) > *((int *)b) ? 1:0 );
}

/*
 * Schedules an event for execution at a future time 
 */
int cs_event_schedule(CsEvent *event, simtime_t time, void *user_data)
{
	CsQueueNode *node;
	CsEventNotice *notice;
	if(! (time < current_time) ){
		node = queue_node_new();
		notice = event_notice_new();
		notice->scheduled_time = time;
		notice->event = event;
		notice->user_data = user_data;
		node->data = notice;
		queue_node_insert_sorted(event_queue,node,num_greater);
	}
	else{	/* scheduled in the past */
		info("Event scheduled in the past, ignoring");
		return -1;
	}
	return 0;
}

/*
 * Schedules an event for execution at a time specified as a relative positive
 * offset to the current time
 */
int cs_event_schedule_relative(CsEvent *event, 
			       simtime_t offset,
			       void *user_data)
{
	if( offset < 0 ){
		info("Event scheduled in the past, ignoring");
		return -1;
	}
	else{
		cs_event_schedule(event, current_time+offset, user_data);
	}
	return 0;
}

	
	
