/* 
 * CSim - Discrete system simulation library
 * 
 * Copyright (C) 2002 Vikas G P <vikasgp@users.sourceforge.net>
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 */

#include <config.h>

#include "csim.h"
#include "entity.h"
#include "util.h"
#include "queue.h"
#include <string.h>


/* Creates a new entity */

CsEntity *cs_entity_new(void)
{
	CsEntity *entity;
	CsQueue *queue;
	entity = (CsEntity *)xmalloc(sizeof(CsEntity));
	queue = (CsQueue *)xmalloc(sizeof(CsQueue));
	entity->attribute_queue = queue;
	return entity;
}

/* Destroys an entity */

void cs_entity_destroy(CsEntity *entity)
{
	queue_destroy(entity->attribute_queue);
	free(entity);
}

/* Adds an attribute to an entity */

/* 
 * Ugly hack. If a function like cs_attribute_set is called internally, it
 * shouldn't output an error message. Therefore this variable
 */
static int internal_call = 0;

int cs_attribute_add(CsEntity *entity ,char *name)
{
	CsAttribute *attribute;
	CsQueueNode *attr_node;
	internal_call = 1;
	if( cs_attribute_set(entity, name, 0) != 0 ){ /* doesn't already exist */
		attr_node = queue_node_new();
		attr_node->key = NULL;
		attribute = (CsAttribute *)xmalloc(sizeof(CsAttribute));
		attribute->name = xstrdup(name);
		attribute->value = 0;
		attr_node->data = (void *)attribute;
		queue_node_append(entity->attribute_queue, attr_node);
		
		internal_call = 0; 
		return 0;
	}
	else{   /* Already exists */
		if(!internal_call){
			info("Attribute specified twice");
		}
		
		internal_call = 0;
		return -1;
	}
}

/* Sets an attribute to a value */

int cs_attribute_set(CsEntity *entity, char *name, int new_value)
{
	CsQueueNode *i = (entity->attribute_queue)->head;
	while( i != NULL ){
		if( !(strcmp( ((CsAttribute *)i->data) -> name, name) ) ){
			((CsAttribute *)i->data)->value = new_value;
			return 0;
		}
		i = i->forward;
	}
	if(!internal_call){
		info("Trying to set non-existent attribute");
	}
	
	internal_call = 0;
	return -1;
}

/* Returns the value of an attribute */

int cs_attribute_get(CsEntity *entity, char *name)
{
        CsQueueNode *i = (entity->attribute_queue)->head;
        while( i != NULL ){
        	if( !(strcmp( ((CsAttribute *)i->data) -> name, name)) ){
                	return ((CsAttribute *)i->data)->value;
		}
		i = i->forward;
	}

#ifdef DIE_ON_NO_ATTR
	die("Trying to get value of non-existent attribute");
#else
	info("Trying to get non-existent attribute");
	return -1;
#endif
}

/* 
 * Adds many attributes to an entity, whose names are supplied in a char **
 * array
 */

void cs_attribute_specify(CsEntity *entity, char **attr_names)
{
	while( attr_names != NULL ){
		cs_attribute_add(entity, *attr_names);
		attr_names++;
	}
}

/* 
 * The sum of create and attribute_specify, useful for creating "temporary
 * entities"
 */

CsEntity *cs_entity_new_with_attributes(char **attr_names)
{
	CsEntity *entity = cs_entity_new();
	cs_attribute_specify(entity, attr_names);
	return entity;
}

