#ifndef __CSIM_H__
#define __CSIM_H__

/* 
 * CSim - Discrete system simulation library
 * 
 * Copyright (C) 2002 Vikas G P <vikasgp@users.sourceforge.net>
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 */

/* 
 * The CSim library provides an embeddable frame-work for discrete system 
 * simulation, using the paradigm adopted by SIMSCRIPT. It is written in C,
 * as opposed to most other such libraries, which are written in C++.
 */

#ifndef NULL
#	define NULL 0
#endif

#include "queue.h"

/*
 * Each entity of a system is represented by a CsEntity. They are created with
 * the function cs_entity_new(...)
 */
typedef struct _CsEntity{
	CsQueue *attribute_queue;	
} CsEntity;

/* 
 * Each event is represented by a CsEvent.
 */
typedef struct _CsEvent{
	int (*event_routine)(void *user_data);
	int count;
} CsEvent;

/* Each set is represented by a CsSet */
typedef struct _CsSet{
        CsQueue *queue;
        int discipline;
        int count;
} CsSet;

/* Data type for clock time */
typedef long simtime_t;

/* Macro for event routines */
#define CS_EVENT_ROUTINE(func)	int (*func)(void *)

/* Synonym for the current time */
#define NOW 	cs_get_clock_time()

/* Checks if set is empty */
#define cs_set_is_empty(set)	cs_set_count_elements((set))

/* Set operations */

#define FIFO	1
#define LIFO	2

/* functions prefixed with cs_ are exported to the outside */

/* 
 * Initializes the library. 
 */
int cs_init(void);

/*
 * Starts the simulation, executing the events until it runs out of events.
 */
int cs_start_simulation(void);

/* Entity functions */

/* 
 * Creates a new entity.
 */
CsEntity *cs_entity_new(void);

/* Destroys an entity */
void cs_entity_destroy(CsEntity *entity);

/* 
 * Adds an attribute to an entity's list of attributes.
 */
int cs_attribute_add(CsEntity *entity, char *name);

/* Set's an attribute to the specified value */
int cs_attribute_set(CsEntity *entity, char *name, int new_value);

/* Returns the value of the specified attribute */
int cs_attribute_get(CsEntity *entity, char *name);

/* 
 * This is a faster way to specify all the attributes of an entity.
 */
void cs_attribute_specify(CsEntity *entity, char **attr_names);

/* To create temp. entities */
CsEntity *cs_entity_new_with_attributes(char **attr_names);

/* Event functions */

/*
 * Creates a new event.
 */
CsEvent *cs_event_new(int (*event_routine)(void *user_data));

/* Get's the event's count */
int cs_event_get_count(CsEvent *event);

/*
 * Schedules an event to executed at the specified time.
 */
int cs_event_schedule(CsEvent *event, simtime_t time, void *user_data);

/* 
 * Same as the cs_event_schedule, but the time is specified as an offset from
 * the current clock time.
 */
int cs_event_schedule_relative(CsEvent *event, simtime_t offset, void *user_data);

/* Set functions */

/* Creates a new set */
CsSet *cs_set_new(int discipline);

/* "Files", i.e., inserts an entity into a set */
void cs_set_file(CsSet *set, CsEntity *entity);

/* Returns the top-most member of a set, and removes it from the set */
CsEntity *cs_set_remove(CsSet *set);

/* Same as above, but doesn't remove it */
CsEntity *cs_set_remove_nodestroy(CsSet *set);

/* Return number of elements in a set */
int cs_set_count_elements(CsSet *set);

/* Destroys a set */
void cs_set_destroy(CsSet *set);


/* Misc. functions */

/* Get's the current time */
simtime_t cs_get_clock_time(void);

#endif		/* __CSIM_H__ */
