//////////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////////
//
// Class          : SimpleDiceEnsemble
//
// Author         : Richard E. Pattis
//                  Computer Science Department
//                  Carnegie Mellon University
//                  5000 Forbes Avenue
//                  Pittsburgh, PA 15213-3891
//                  e-mail: pattis@cs.cmu.edu
//
// Maintainer     : Author
//
//
// Description:
//
//   Objects constructed from the SimpleDiceEnsemble class act as
// collections of dice. The number of dice in an ensemble (and the
// number of sides of each die) can be controlled by the programmer.   
//
// Future Plans   : implement min/max methods
//
// Program History:
//   5/18/01: R. Pattis - Operational for 15-100
//   8/20/01: R. Pattis - JavaDoc comments added
//
//
//////////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////////


/** 
 * Objects constructed from the <code>SimpleDiceEnsemble</code> class act
 *   as collections of dice.
 * The number of dice in an ensemble (and the number of sides of each die)
 *   can be controlled by the programmer.
 * The class models the basic operations need to roll the dice and determine
 *   the number of pips showing -but only for the for the entire ensemble
 * The regular <code>DiceEnsemble</code> class also stores the individual
 *   pips for each die (retrievable via the <code>getPips</code> method
 *   that is not present here).   
 * 
 * @author Richard E. Pattis (Computer Science Department, Carnegie Mellon)
*/
public class SimpleDiceEnsemble { 


	/** 
	 * Constructs a <code>DiceEnsemble</code> object, specifying the number
	 *   of dice and the number of sides per die.
	 *
	 * @param  numberOfDice specifies the number of dice in the ensemble
	 * @param  sidesPerDie specifies the number of sides on each and every
	 *           die in the ensemble (the pips showing are 1, 2, 3, ...
	 *           <code>sidesPerDie</code>)
	 *
	 * @throws IllegalArgumentException if either parameter is less than 1
	*/
  public SimpleDiceEnsemble (int numberOfDice, int sidesPerDie)
      throws IllegalArgumentException
  {
    if (numberOfDice < 1)
      throw new IllegalArgumentException("SimpleDiceEnsemble constructor: Number of dice ("+numberOfDice+") < 1"); 
    if (sidesPerDie < 1)
      throw new IllegalArgumentException("SimpleDiceEnsemble constructor: Sides per die ("+sidesPerDie+") < 1"); 

    this.numberOfDice = numberOfDice;
    this.sidesPerDie  = sidesPerDie;
    //rollCount: see declaration for implicit initializaton to 0
    //pipCount and allSame indeterminate until roll
  }
  
  
  
	/** 
	 * Constructs a <code>DiceEnsemble</code> object, consisting of
	 *   two dice, each having six sides: a standard ensemble.
	*/
  public SimpleDiceEnsemble ()
  {this(2,6);}
  
  
  
	/** 
	 * Returns a random number in the specified range
	 *   <code>[1..sidesPerDie]</code>.
	 *
	 * @return a random number in the specified range
	 *   <code>[1..sidesPerDie]</code>
	*/
  private int randomDie ()
  {return (int)Math.round(Math.random()*sidesPerDie + .5);}

  
  
	/** 
	 * Returns this <code>SimpleDiceEnsemble</code> after rolling every die in it.
	 * By using the return type <code>SimpleDiceEnsemble</code> instead of
	 *  <code>void</code>, we can "cascade" method calls, writing expressions
	 *  such as: <code>d.roll().getPipSum()</code> instead of writing roll as
	 *  its own statement.
	 *
	 * @return the rolled dice ensemble
	*/
  public SimpleDiceEnsemble roll ()
  {
    rollCount++;
    int firstThrow = randomDie();
    pipSum  = firstThrow;
    allSame = true;
    for (int i=2; i<=numberOfDice; i++) {
      int nextThrow = randomDie();
      pipSum += nextThrow;
      allSame = allSame && (nextThrow == firstThrow);
    }
    return this;
  }

  
  
	/** 
	 * Returns the number of dice in this ensemble (set at construction).
	 *
	 * @return the number of dice in this ensemble
	*/
  public int getNumberOfDice ()
  {return numberOfDice;}
  
  
  
	/** 
	 * Returns the number of sides on each die in this ensemble (set at
	 *   construction).
	 *
	 * @return the number of sides on each die in this ensemble
	*/
  public int getSidesPerDie ()
  {return sidesPerDie;}
  
  

	/** 
	 * Returns the sum of all the pips showing in this ensemble (changed
	 *   each time the ensemble is rolled).
	 *
	 * @throws IllegalStateException if this ensemble has not been rolled
	 *
	 * @return the sum of all the pips showing in this ensemble
	*/
  public int getPipSum ()
    throws IllegalStateException
  {
    if (rollCount == 0)
      throw new IllegalStateException("getPipSum called, but dice not rolled");
 
    return pipSum;
  }
  
  

	/** 
	 * Returns the number of times the dice in this ensemble have been
	 *   rolled (incremented by 1 each time the ensemble is rolled).
	 *
	 * @return the number of times the dice in this ensemble have been
	 *         rolled
	*/
  public int getRollCount ()
  {return rollCount;}
  
  

	/** 
	 * Returns whether the pips showing for each die are all the same.
	 *
	 * @throws IllegalStateException if this ensemble has not been rolled
	 *
	 * @return whether the pips showing for each die are all the same;
	 *         for games like Monopoly, which use two dice, it is sometimes
	 *         important to know whether the user rolled doubles.
	*/
  public boolean allSame ()
    throws IllegalStateException
  {
    if (rollCount == 0)
      throw new IllegalStateException("allSame called, but dice not rolled");

    return allSame;
  }
  
  
  
	/** 
	 * Returns a <code>String<code> representation of the state of this ensemble.
	 *
	 * @return (1) the number of dice in the ensemble, (2) the number of
	 *         sides on each die, (3) the number of times the dice ensemble
	 *         has been rolled, (4) the pips sum for the current roll,
	 *         and (5) whether all the dice showed the same number of
	 *         pips on the current throw
	*/
  public String toString ()
  {
    return "DiceEnsemble[numberOfDice="+getNumberOfDice()+"/"+
           "sidesPerDie="+getSidesPerDie()+"/"+
           "rollCount="+getRollCount()+"/"+
           "pipSum="+pipSum+"/" +
           "allSame="+allSame+"]";
  }
  
  
  
  //No JavaDoc comments for private instance variables; they are well-named
  
  private int     numberOfDice;
  private int     sidesPerDie;
  private int     rollCount;
  private int     pipSum;
  private boolean allSame;
}




