package edu.uci.ics.luci.projects.nomatic.GPS;

import java.io.IOException;
import java.io.InputStream;

import javax.microedition.io.Connector;
import javax.microedition.io.HttpConnection;
import javax.microedition.io.StreamConnection;

import edu.uci.ics.luci.projects.nomatic.Bluetooth.Bluetooth;

public class GPSComponent implements Runnable {
	/** Configuration options * */
	/** configuration takes on one of the CONFIG* values * */
	private int configuration;

	/*
	 * This configuration scans for bluetooth devices, and then discovers the
	 * services on one of them
	 */
	public static final int CONFIG_TEST_BLUETOOTH = 0;

	/* This configuration gets GPS reading from a file on the Internet */
	public static final int CONFIG_USE_NETWORK_GPS_PROXY = 1;

	/* This configuration gets GPS readings from a blueooth GPS */
	public static final int CONFIG_USE_BLUETOOTH_GPS = 2;

	public boolean GPSComponentSetupDone;

	/** Logging fields * */
	private static final int LOG_INFORMATION = 10;

	private static final int LOG_EXCEPTION = 0;

	public static final int USE_FIRST = NMEASentence.USE_FIRST;
	public static final int USE_LAST = NMEASentence.USE_LAST;

	/** Any messages whose importance is less than or equal to this get recorded* */
	private int LOG_MESSAGE_LEVEL = 10;

	/** Log Message Storage * */
	private StringBuffer logMessage;

	/** Bluetooth related parameters * */
	private Bluetooth blueTooth;

	private int bluetoothMatch;

	private String bluetoothSignature;

	/** GPS Network proxy fields * */
	private String GPS_DATA_URL;

	private StringBuffer GPSBuffer;

	private Thread thread;

	public synchronized boolean isGPSComponentSetupDone() {
		if (getBlueTooth() == null) {
			return (false);
		}
		if (!getBlueTooth().isBluetoothSetupDone()) {
			return (false);
		} else {
			return (true);
		}
	}

	private synchronized Thread getThread() {
		return thread;
	}

	private synchronized void setThread(Thread thread) {
		this.thread = thread;
	}

	private synchronized void addLogMessage(String m, int level) {
		if (level <= LOG_MESSAGE_LEVEL) {
			getLogMessage().append(m);
		}
	}

	/**
	 * This method sets the input stream that the GPS Component takes it's data
	 * from. It blocks until bluetooth setup is done incase the bluetooth setup
	 * finds an inputstream asynchronously. The intent is that this method would
	 * over ride what the bluetooth setup finds as an input stream
	 */
	public void setInputStream(InputStream inputStream) {
		while (!isGPSComponentSetupDone()) {
			try {
				addLogMessage("Waiting for Bluetooth Setup\n", LOG_INFORMATION);
				Thread.sleep(1000);
			} catch (InterruptedException e) {
			}
		}
		getBlueTooth().setInputStream(inputStream);
	}

	public synchronized InputStream getInputStream() {
		if (isGPSComponentSetupDone()) {
			return (getBlueTooth().getInputStream());
		} else {
			return null;
		}
	}

	private synchronized StringBuffer getLogMessage() {
		return logMessage;
	}

	private synchronized void logMessageClear() {
		logMessage.delete(0, logMessage.length());
	}

	public synchronized String getLogMessage(boolean clear) {
		String x = logMessage.toString();
		if (clear) {
			logMessageClear();

		}
		return x;
	}

	private synchronized void setLogMessage(StringBuffer logMessage) {
		this.logMessage = logMessage;
	}

	public synchronized Bluetooth getBlueTooth() {
		return blueTooth;
	}

	private synchronized void setBlueTooth(Bluetooth blueTooth) {
		this.blueTooth = blueTooth;
	}

	public synchronized StringBuffer getGPSBuffer() {
		return GPSBuffer;
	}

	private synchronized void setGPSBuffer(StringBuffer buffer) {
		GPSBuffer = buffer;
	}

	private synchronized void GPSBufferClear(int start, int end) {
		GPSBuffer.delete(start,end);
	}

	private synchronized void GPSBufferAppend(char c) {
		GPSBuffer.append("" + c);
	}

	/**
	 * This is the public interface to GPSBuffer. Should be preceeded by a call
	 * to fillGPSBuffer if there is to be anything in it. This just returns the
	 * raw data found on the input stream.
	 * 
	 * @param clearAfterRead
	 *            if the data returned should be cleared from the read buffer
	 *            then this should be true. If it isn't cleared, then the same
	 *            data will be included in subsequent calls.
	 * @return All of the recently received GPS data
	 */
	public synchronized String getGPSBuffer(String sig, boolean clearAfterRead,
			int firstOrLast) {
		int start = -1;
		int end = -1;
		String temp = getGPSBuffer().toString();
		String ret = null;
		
		if (firstOrLast == GPSComponent.USE_FIRST) {
			start = temp.indexOf(sig);
		} else if (firstOrLast == GPSComponent.USE_LAST) {
			end = temp.lastIndexOf('*',temp.length()-3);
			int pointer= temp.indexOf(sig);
			while((pointer != -1)&&(pointer<end)){
				start=pointer;
				pointer = temp.indexOf(sig,pointer+1);
			}
		}
		if (start != -1) {
			end = temp.indexOf("*", start);
			if (end != -1) {
				if (getGPSBuffer().length() >= end + 3) {
					ret = (temp.substring(start, end + 3));
				}
				else{
					end = -1;
				}
			}
		}
		if (clearAfterRead) {
			if (end != -1) {
				GPSBufferClear(0, end + 3);
			}
		}
		return(ret);
	}

	/**
	 * This is another public interface to GPSBuffer, but it returns a NMEA
	 * Sentence instead of a raw String. If the configuration indicates that the
	 * data is coming from a bluetooth GPS receiver this will return the most
	 * recent NMEA sentence that matches the sig. If the configuration indicates
	 * that the data is coming from some other proxy source, like a file, then
	 * this will return the next NMEA sentence that matches the sig. If
	 * clearAfterRead is false, then subsequent calls will consider the existing
	 * data again in figuring out what to return.
	 * 
	 * @param sig
	 * @param clearAfterRead
	 * @return
	 */
	public NMEASentence getGPSBuffer(String sig, boolean clearAfterRead) {
		if (sig.compareTo("$GPRMC") != 0) {
			return (null);
		} else {
			String s;
			if (getConfiguration() == CONFIG_USE_BLUETOOTH_GPS) {
				s = getGPSBuffer(sig,clearAfterRead, NMEASentence.USE_LAST);
				return (new NMEASentence(s, NMEASentence.USE_LAST));
			} else if (getConfiguration() == CONFIG_USE_NETWORK_GPS_PROXY) {
				s = getGPSBuffer(sig,clearAfterRead, NMEASentence.USE_FIRST);
				return (new NMEASentence(s, NMEASentence.USE_FIRST));
			} else if (getConfiguration() == CONFIG_TEST_BLUETOOTH) {
				/* This option doesn't really make sense */
				s = getGPSBuffer(sig,clearAfterRead, NMEASentence.USE_LAST);
				return (new NMEASentence(s, NMEASentence.USE_LAST));
			} else {
				return (null);
			}
		}
	}

	private synchronized String getGPS_DATA_URL() {
		return GPS_DATA_URL;
	}

	private synchronized void setGPS_DATA_URL(String gps_data_url) {
		GPS_DATA_URL = gps_data_url;
	}

	private synchronized int getBluetoothMatch() {
		return bluetoothMatch;
	}

	private synchronized void setBluetoothMatch(int bluetoothMatch) {
		this.bluetoothMatch = bluetoothMatch;
	}

	private synchronized String getBluetoothSignature() {
		return bluetoothSignature;
	}

	private synchronized void setBluetoothSignature(String bluetoothSignature) {
		this.bluetoothSignature = bluetoothSignature;
	}

	private synchronized int getConfiguration() {
		return configuration;
	}

	private synchronized void setConfiguration(int configuration) {
		this.configuration = configuration;
	}

	/**
	 * This method attempts to loads some data from the GPS source. There is an
	 * upper limit to the amount of data that is loaded.
	 * 
	 * returns: the number of characters moved into Buffer
	 */
	public synchronized int fillGPSBuffer() {
		int upperlimit = 50000;
		int count = 0;

		if (isGPSComponentSetupDone()) {
			/*
			 * inputStream should be set by the Bluetooth initialization
			 * process, or the GPS Component initialization process depending on
			 * whether the data source is bluetooth or network
			 */
			if (getInputStream() != null) {
				try {
					/** Fill up my local buffer * */
					int avail = getInputStream().available();
					while ((count < upperlimit) && (avail > 0)) {
						count += avail;
						while (avail > 0) {
							GPSBufferAppend((char) getInputStream().read());
							avail--;
						}
						avail = getInputStream().available();
					}
				} catch (Exception e) {
					addLogMessage("in fillGPSBuffer " + e.toString() + "\n",
							LOG_EXCEPTION);
				}
			}
		}
		return (count);
	}

	/**
	 * This method checks for the presence of an NMEA sentence in the GPS
	 * buffer. This means it looks for the String sig followed by a "*" and two
	 * more characters
	 * 
	 * @param sig
	 */
	public synchronized boolean checkGPSBufferForNMEASentence(String sig) {
		/** Check if local buffer has a NMEA sentence * */
		int mark;
		/* If we see the signature */
		if ((mark = getGPSBuffer().toString().indexOf(sig)) != -1) {
			int mark2;
			/* If there is a checksum after the signature */
			if ((mark2 = getGPSBuffer().toString().indexOf("*", mark + 1)) != -1) {
				if (getGPSBuffer().toString().substring(mark2 + 1).length() >= 2) {
					return (true);
				}
			}
		}
		return (false);
	}

	public GPSComponent(int configuration) {
		this(
				configuration,
				new String(
						"http://www.ics.uci.edu/~djp3/classes/2007_04_02_INF132/code/GPSDATA01.txt"));
	}

	public GPSComponent(int configuration, String GPS_DATA_URL) {
		/* Initialize general fields */
		setLogMessage(new StringBuffer());
		blueTooth = null;
		setGPSBuffer(new StringBuffer());
		setConfiguration(configuration);
		setGPS_DATA_URL(GPS_DATA_URL);

		/* Initialize bluetooth parameters */
		if (getConfiguration() == CONFIG_TEST_BLUETOOTH) {
			setBluetoothMatch(Bluetooth.NEVER_GET_TO_MATCHING);
			setBluetoothSignature(new String(""));
		}
		if (getConfiguration() == CONFIG_USE_NETWORK_GPS_PROXY) {
			setBluetoothMatch(Bluetooth.NEVER_GET_TO_MATCHING);
			setBluetoothSignature(new String(""));
		}
		if (getConfiguration() == CONFIG_USE_BLUETOOTH_GPS) {
			setBluetoothMatch(Bluetooth.MATCH_SIGNATURE);
			setBluetoothSignature(new String("BTGPS 2285B9"));
		}

		setThread(new Thread(this));
		getThread().start();
	}

	public void run() {

		/**
		 * Launch Bluetooth stuff, this is asynchronous, so it returns before it
		 * is done *
		 */
		launchBluetooth();
		/**
		 * Launch Network GPS component, returns immediately if its not
		 * applicable to the configuration *
		 */
		launchNetworkGPS();

	}

	/**
	 * After bluetooth setup is done and completing this method
	 * GPSComponenet::inputStream will be set, whether by bluetooth or otherwise
	 * 
	 * 
	 */

	private void launchNetworkGPS() {
		if (getConfiguration() == CONFIG_USE_NETWORK_GPS_PROXY) {
			try {
				addLogMessage("About to open\n", LOG_INFORMATION);
				StreamConnection c;
				c = (StreamConnection) Connector.open(getGPS_DATA_URL(),
						Connector.READ, false);
				addLogMessage("About to get\n", LOG_INFORMATION);
				HttpConnection c2 = (HttpConnection) c;
				addLogMessage("URL: " + c2.getURL() + "\n", LOG_INFORMATION);
				addLogMessage("Protocol: " + c2.getProtocol() + "\n",
						LOG_INFORMATION);
				addLogMessage("Host: " + c2.getHost() + "\n", LOG_INFORMATION);
				addLogMessage("File: " + c2.getFile() + "\n", LOG_INFORMATION);
				addLogMessage("Ref: " + c2.getRef() + "\n", LOG_INFORMATION);
				addLogMessage("Query: " + c2.getQuery() + "\n", LOG_INFORMATION);
				addLogMessage("Port: " + c2.getPort() + "\n", LOG_INFORMATION);
				addLogMessage("Method: " + c2.getRequestMethod(),
						LOG_INFORMATION);
				InputStream in = c.openInputStream();
				/** Override bluetooth stream with our stream * */
				setInputStream(in);

				addLogMessage("Input Stream = " + in.toString() + "\n",
						LOG_INFORMATION);

				addLogMessage("\nResponseCode: " + c2.getResponseCode()
						+ "\nResponseMessage:" + c2.getResponseMessage()
						+ "\nContentLength: " + c2.getLength()
						+ "\nContentType: " + c2.getType()
						+ "\nContentEncoding: " + c2.getEncoding()
						+ "\nContentExpiration: " + c2.getExpiration()
						+ "\nDate: " + c2.getDate() + "\nLast-Modified: "
						+ c2.getLastModified() + "\n\n", LOG_INFORMATION);
			} catch (IOException error) {
				System.out.println(error.toString());
				error.printStackTrace();
			}
		} else {
			addLogMessage("Wrong configuration:" + getConfiguration(),
					LOG_INFORMATION);
		}
	}

	private void launchBluetooth() {
		addLogMessage("Launching Bluetooth Component\n", LOG_INFORMATION);

		try {
			setBlueTooth(new Bluetooth(getBluetoothMatch(),
					getBluetoothSignature()));
			addLogMessage(getBlueTooth().getLogMessage(), LOG_INFORMATION);

		} catch (Exception e) {
			addLogMessage(
					"Exception in Bluetooth constructor: " + e.toString(),
					LOG_EXCEPTION);
		}
		addLogMessage("Done Launching Bluetooth Component\n", LOG_INFORMATION);
	}

}
